import Foundation
import SwiftUI

/// Type of a widget
typealias WidgetID = String

/// Represents an home widget in a list.
protocol Widget: Identifiable {
    /// Must be a `SwiftUI.View`
    associatedtype ViewBody: View
    associatedtype Configuration

    var id: WidgetID { get }
    func make(configuration: Configuration) -> ViewBody
}

extension Widget {
    var id: WidgetID {
        String(describing: Self.self)
    }
}

protocol WidgetFactory {
    var id: WidgetID { get }
    var contentView: AnyView { get }
}

struct AnyWidget: Identifiable, WidgetFactory {
    private let widget: any Identifiable
    let id: WidgetID
    private let make: () -> AnyView

    init<WidgetType, Configuration>(_ widget: WidgetType, configuration: Configuration) where WidgetType: Widget, WidgetType.Configuration == Configuration {
        self.widget = widget
        self.id = widget.id
        self.make = {
            AnyView(widget.make(configuration: configuration))
        }
    }

    var contentView: AnyView {
        make()
    }
}
